/* -*- mode: c++; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <cerrno>
#include <cmath>

#include "util.h"


namespace
{
    template<typename T>
    T
    _deltatime(const struct timespec *base, size_t nmemb)
    {
        size_t i;
        T t, sum_x, sum_xi;

        if (nmemb <= 1)
            return (T(0));

        sum_x = sum_xi = 0;
        for (i = 1; i < nmemb; i++) {
            t = difftime(base[i].tv_sec, base[0].tv_sec) +
                1e-9 * (base[i].tv_nsec - base[0].tv_nsec);
            sum_x += t;
            sum_xi += t * i;
        }

        t = T(6) / (nmemb * (nmemb + 1)) *
            (T(2) * sum_xi / (nmemb - 1) - sum_x);
        return (t);
    }
}


double
deltatime(const struct timespec *base, size_t nmemb)
{
    return (_deltatime<double>(base, nmemb));
}


float
deltatimef(const struct timespec *base, size_t nmemb)
{
    return (_deltatime<float>(base, nmemb));
}


namespace
{
    template<typename T>
    T
    _ht2wavelength(T ht)
    {
        /* The speed of light (e8*m/s), the elementary charge
         * (e-19*C), Planck constant (e-34*Js), and the electron rest
         * mass (e-31*kg) are from http://www.wikipedia.org.  All
         * constants are defined by their "normalized" significand
         * only, because otherwise the product of e.g. m and e would
         * overflow the exponent of the IEEE single-precision binary
         * floating-point format.
         */
        const T c = 2.99792458;
        const T e = 1.602176565;
        const T h = 6.62606957;
        const T m = 9.10938215;

        T k2, p2;

        if (ht <= T(0)) {
            errno = EDOM;
            return (0);
        }


        /* The classical squared momentum of an electron accelerated
         * by an electrostatic potential ht.
         */
        p2 = T(2e-4) * m * e * ht;


        /* The relativistic correction factor to the classical squared
         * momentum.
         */
        k2 = T(1) + e * ht / (T(2e4) * m * c * c);

        return (T(1e-11) * h / std::sqrt(p2 * k2));
    }
}


double
ht2wavelength(double ht)
{
    return (_ht2wavelength(ht));
}


float
ht2wavelengthf(float ht)
{
    return (_ht2wavelength(ht));
}
