/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <stdarg.h>
#include <stdlib.h>

#include <ctype.h>
#include <errno.h>
#include <float.h>
#include <math.h>
#include <string.h>

#include "frame.h"


void
frame_free(struct frame *frame)
{
    if (frame == NULL)
        return;

    if (frame->id != NULL)
        free(frame->id);
    if (frame->raster != NULL)
        free(frame->raster);

    free(frame);
}


struct frame *
frame_new()
{
    struct frame *frame;

    frame = malloc(sizeof(struct frame));
    if (frame == NULL)
        return (NULL);

    frame->tv.tv_sec = 0;
    frame->tv.tv_nsec = 0;

    frame->beam_center[0] = NAN;
    frame->beam_center[1] = NAN;

    frame->pixel_size[0] = 0;
    frame->pixel_size[1] = 0;

    frame->binning[0] = 1;
    frame->binning[1] = 1;

    frame->id = NULL;
    frame->raster = NULL;

    frame->height = 0;
    frame->width = 0;

    frame->distance = 0;
    frame->exposure = 0;
    frame->osc_range = 0;
    frame->osc_start = 0;
    frame->wavelength = 0;

    return (frame);
}


int
frame_flip(struct frame *frame)
{
    uint16_t *dst;
    size_t i, j;

    dst = calloc(frame->width * frame->height, sizeof(uint16_t));
    if (dst == NULL)
        return (-1);

    for (i = 0; i < frame->height; i++) {
        for (j = 0; j < frame->width; j++) {
            dst[frame->width * (frame->height - 1 - i) + j] =
                frame->raster[frame->width * i + j];
        }
    }

    free(frame->raster);
    frame->raster = dst;
    return (0);
}


int
frame_rot90(struct frame *frame, int k)
{
    uint16_t *dst;
    size_t i, j;
    ssize_t inc, ld, off;

    /* The row-major source array is always read from left to right,
     * and from top to bottom.  Calculate the increment, the leading
     * dimension, and the offset for the destination array, just like
     * in the BLAS code.
     */
    switch (k % 4) {
    case 0:
        inc = 1;
        ld = frame->width;
        off = 0;
        break;

    case -3:
    case +1:
        inc = -frame->height;
        ld = 1;
        off = frame->height * (frame->width - 1);
        break;

    case -2:
    case +2:
        inc = -1;
        ld = -frame->width;
        off = frame->width * frame->height - 1;
        break;

    case -1:
    case +3:
        inc = frame->height;
        ld = -1;
        off = frame->height - 1;
        break;

    default:
        /* NOTREACHED */
        errno = EDOM;
        return (-1);
    }

    /* The copy itself cannot be performed in place, so allocate new
     * space for the rotated image before copying the data there.
     * Swap width and height unless rotating an even multiple of 90
     * degrees.
     */
    dst = calloc(frame->width * frame->height, sizeof(uint16_t));
    if (dst == NULL)
        return (-1);

    for (i = 0; i < frame->height; i++) {
        for (j = 0; j < frame->width; j++)
            dst[ld * i + inc * j + off] = frame->raster[frame->width * i + j];
    }

    free(frame->raster);
    frame->raster = dst;
    if (k % 2 != 0) {
        i = frame->height;
        frame->height = frame->width;
        frame->width = i;
    }

    return (0);
}


/**
 * The _asncatf() function appends a string to a dynamically allocated
 * string of size @p size pointed to by @p ret.  The string is defined
 * by @p format and @p ... according to printf(3).  The destination
 * string @p ret is dynamically grown to fit @p format, but at least
 * by 1024 characters.
 *
 * @param ret    String to append to.  If @p ret is @c NULL, it will
 *               be dynamically allocated.
 * @param size   Length of block allocated for @p ret.  This is
 *               ignored if @p ret is @c NULL.
 * @param format Format string according to printf(3)
 * @param ...    Format arguments according to printf(3)
 * @return       The number of characters written, not including the
 *               trailing @c NULL used to end output to strings.  If
 *               an output or encoding error occurs, a value of -1 is
 *               returned.
 */
static int
_asncatf(char **ret, size_t *size, const char *format, ...)
{
    va_list ap;
    void *p;
    size_t BLOCKSIZE = 1024, len;
    int d;

    /* If called with an empty string, i.e. if *ret is NULL, the first
     * iteration will allocate an initial block.  Because the format
     * is constant throughout the iteration, the loop is guaranteed to
     * terminate.
     */
    if (*ret == NULL)
        len = *size = 0;
    else
        len = strlen(*ret);
    for ( ; ; ) {
        /* Try to append the formatted string and exit successfully if
         * all of it fit.
         */
        va_start(ap, format);
        d = vsnprintf(*ret + len, *size - len, format, ap);
        va_end(ap);
        if (d < 0)
            return (-1);
        if (d + 1 <= *size - len)
            return (d);

        /* Make sure BLOCKSIZE is big enough to accommodate the
         * overhang on the next iteration.
         */
        if (BLOCKSIZE < d + 1 - (*size - len))
            BLOCKSIZE = d + 1 - (*size - len);

        /* The attempted concatenation resulted in an overhang.  Grow
         * the block and try again.
         */
        p = realloc(*ret, *size + BLOCKSIZE);
        if (p == NULL)
            return (-1);
        *ret = p;
        *size += BLOCKSIZE;
    }

    /* NOTREACHED
     */
    return (-1);
}


/* In order for the same binary representation to be recreated when
 * the number is read back, single and double precision real numbers
 * must be output with 9 and 17 significant decimal digits,
 * respectively (Kahan, 1997).
 *
 * XXX This, together with the corresponding read() function, is maybe
 * better placed in a separate smv.c file, especially once the
 * experimental/cbf branch is merged.
 */
int
frame_write(const struct frame *frame, FILE *stream)
{
    float bp[2];
    char *buf, *endptr, *header;
    void *p;
    struct tm tm;
    size_t len, size;
    int d, padding, ret;

    buf = NULL;
    header = NULL;
    size = 0;


    /* Output the beam center, if known, along the orthogonal axes,
     * and the binning factors.
     */
    if (isfinite(frame->beam_center[1])) {
        if (_asncatf(&header, &size,
                     "BEAM_CENTER_X=%-.9g;\n",
                     frame->beam_center[1]) <= 0) {
            goto exit_error;
        }
    }
    if (isfinite(frame->beam_center[0])) {
        if (_asncatf(&header, &size,
                     "BEAM_CENTER_Y=%-.9g;\n",
                     frame->beam_center[0]) <= 0) {
            goto exit_error;
        }
    }
    if (_asncatf(&header, &size,
                 "BIN=%dx%d;\n",
                 frame->binning[1], frame->binning[0]) <= 0) {
        goto exit_error;
    }


    /* The byte order of the output file is always the native byte
     * order of the host.
     *
     * XXX The code below probably only works on GCC-like compilers.
     * This and other instances should be fixed once the suite is
     * autoconfiscated.
     */
#ifdef __BIG_ENDIAN__
    if (_asncatf(&header, &size,
                 "BYTE_ORDER=big_endian;\n") <= 0) {
        goto exit_error;
    }
#else
    if (_asncatf(&header, &size,
                 "BYTE_ORDER=little_endian;\n") <= 0) {
        goto exit_error;
    }
#endif


    /* Write the UTC construction time (again) in "standard" format.
     * This format is 24 characters long in the standard locale, not
     * counting the terminating NULL character.  Because it may be
     * longer in different locales, grow the buffer until it fits.  If
     * the date does not fit into a 512-character buffer, fail with
     * EMSGSIZE.
     */
    if (gmtime_r(&frame->tv.tv_sec, &tm) == NULL)
        goto exit_error;
    for (len = 24 + 1; ; len += 24 + 1) {
        p = realloc(buf, len * sizeof(char));
        if (p == NULL)
            goto exit_error;
        buf = p;

        if (strftime(buf, len, "%a %b %e %H:%M:%S %Y", &tm) > 0)
            break;

        if (len > 512) {
            errno = EMSGSIZE;
            goto exit_error;
        }
    }
    if (_asncatf(&header, &size,
                 "DATE=%s;\n",
                 buf) <= 0) {
        goto exit_error;
    }


    /* DETECTOR_SN is required for imginfo from Global Phasing's
     * processing tools.  MOSFLM breaks if it is set to the empty
     * string, and iotbx requires it to be either an integer, "None",
     * or "unknown".
     */
    if (frame->id != NULL &&
        strlen(frame->id) > 0 &&
        strtol(frame->id, &endptr, 10) > 0 &&
        endptr[0] == '\0') {
        if (_asncatf(&header, &size,
                     "DETECTOR_SN=%s;\n",
                     frame->id) <= 0) {
            goto exit_error;
        }
    } else {
        if (_asncatf(&header, &size,
                     "DETECTOR_SN=unknown;\n") <= 0) {
            goto exit_error;
        }
    }


    /* All images contained in a frame structure are two-dimensional.
     */
    if (_asncatf(&header, &size,
                 "DIM=2;\n"
                 "DISTANCE=%-.9g;\n",
                 frame->distance) <= 0) {
        goto exit_error;
    }


    /* Oscillation start and oscillation range.  MOSFLM ignores
     * OSC_START in favor of PHI unless an OSC_AXIS is present, in
     * which case it reads OSC_START instead of PHI.
     */
    if (_asncatf(&header, &size,
                 "OSC_RANGE=%-.9g;\n"
                 "OSC_START=%-.9g;\n"
                 "PHI=%-.9g;\n",
                 frame->osc_range, frame->osc_start, frame->osc_start) <= 0) {
        goto exit_error;
    }


    /* This at least agrees with dxtbx's idea of width, height, SIZE1
     * and SIZE2.
     */
    if (_asncatf(&header, &size,
                 "SIZE1=%zd;\n"
                 "SIZE2=%zd;\n",
                 frame->width, frame->height) <= 0) {
        goto exit_error;
    }


    /* Pixels must be square, to within machine epsilon, or this
     * function will fail with EINVAL.  This implementation is
     * inspired by Knuth (1997) "The Art of Computer Programming,
     * Volume 2: Seminumerical Algorithms".
     */
    bp[0] = frame->binning[0] * frame->pixel_size[0];
    bp[1] = frame->binning[1] * frame->pixel_size[1];
    if (fabs(bp[0] - bp[1]) / (bp[0] < bp[1] ? bp[0] : bp[1]) > FLT_EPSILON) {
        errno = EINVAL;
        goto exit_error;
    }
    if (_asncatf(&header, &size,
                 "PIXEL_SIZE=%-.9g;\n"
                 "TIME=%-.9g;\n"
                 "WAVELENGTH=%-.9g;\n",
                 bp[0],
                 frame->exposure,
                 frame->wavelength) <= 0) {
        goto exit_error;
    }


    /* The theta-angle is fixed at zero for frame structures.  In SMV
     * parlance, an unsigned_short is 16 bits wide--no other types are
     * known to be in use.
     */
    if (_asncatf(&header, &size,
                 "TWOTHETA=0;\n"
                 "TYPE=unsigned_short;\n"
                 "}\n") <= 0) {
        goto exit_error;
    }


    /* Construct the HEADER_BYTES item, and fail with EMSGSIZE if it
     * does not fit into a 512-character buffer.
     */
    d = padding = 0;
    for ( ; ; ) {
        /* Write a tentative item and count its length.
         */
        ret = snprintf(buf, len,
                       "{\n"
                       "HEADER_BYTES=%zd;\n",
                       d + strlen(header) + padding);
        if (ret < 0)
            goto exit_error;
        if (ret + 1 <= len) {
            /* Iterate as long as the length of the output and the
             * HEADER_BYTES item's contribution differ.
             */
            if (d != ret) {
                d = ret;
                continue;
            }

            /* It appears MOSFLM requires HEADER_BYTES to be an
             * integer multiple of 512.  Pad if necessary and iterate.
             */
            if ((d + strlen(header) + padding) % 512 != 0) {
                padding = 512 - (d + strlen(header)) % 512;
                continue;
            }
            break;
        }

        /* Grow the buffer as necessary.
         */
        p = realloc(buf, (ret + 1) * sizeof(char));
        if (p == NULL)
            goto exit_error;
        buf = p;
        len = ret + 1;

        if (len > 512) {
            errno = EMSGSIZE;
            goto exit_error;
        }
    }


    /* Write the whole caboodle, the HEADER_BYTES item, the header,
     * the padding, and the data.
     */
    if (fprintf(stream, "%s%s", buf, header) != strlen(buf) + strlen(header))
        goto exit_error;
    if (fprintf(stream, "%*.s", padding, "") != padding)
        goto exit_error;
    len = frame->width * frame->height;
    if (fwrite(frame->raster, sizeof(uint16_t), len, stream) != len)
        goto exit_error;
    ret = 0;
    goto exit_success;

exit_error:
    ret = -1;

exit_success:
    if (buf != NULL)
        free(buf);
    if (header != NULL)
        free(header);
    return (ret);
}
