/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef FRAME_H
#define FRAME_H 1

#ifdef __cplusplus
#  define FRAME_BEGIN_C_DECLS extern "C" {
#  define FRAME_END_C_DECLS   }
#else
#  define FRAME_BEGIN_C_DECLS
#  define FRAME_END_C_DECLS
#endif

FRAME_BEGIN_C_DECLS

/**
 * @file frame.h
 * @brief XXX
 *
 * XXX
 */

#include <stdio.h>
#include <stdint.h>

#include <time.h>


/**
 * The frame structure consists of an image: a two-dimensional array,
 * or raster, of pixel intensities and associated metadata.  The
 * raster is interpreted as row-major: the fast-changing direction is
 * assumed to run horizontally from left to right, while the
 * slow-changing direction runs vertically from top to bottom.  The
 * content of this structure mirrors the limitations of the SMV
 * format.  In particular, pixel intensities are represented as 16-bit
 * unsigned integers and pixels are assumed to be square.
 */
struct frame
{
    /**
     * @brief Timestamp, s and ns elapsed since midnight, 1 January
     *        1970 UTC (Unix time)
     */
    struct timespec tv;

    /**
     * @brief The beam center in the horizontal (fast-changing) and
     *        vertical (slow-changing) directions in mm
     *
     * Any non-finite value indicates that the beam center in the
     * given direction is unknown.  In MOSFLM and XDS @c BEAM_CENTER_X
     * and @c BEAM_CENTER_Y define the distance of beam center from
     * the origin in mm in the slow- and fast-changing directions,
     * respectively.
     */
    float beam_center[2];

    /**
     * @brief The pixel size in the horizontal (fast-changing) and
     *        vertical (slow-changing) directions in mm
     *
     * XXX The SMV format really only supports a 1D-pixel size for
     * square pixels.
     */
    float pixel_size[2];

    /**
     * @brief The binning factor in the horizontal (fast-changing) and
     *        vertical (slow-changing) directions
     *
     * XXX Should probably be one-dimensional, because pixels in SMV
     * must be square.
     */
    unsigned int binning[2];

    /**
     * @brief Detector identifier as a NULL-terminated string
     *
     * XXX This is output as @c DETECTOR_SN.  But what is the
     * analogous item in DM4 and CBF?
     */
    char *id;

    /**
     * @brief Image raster in row-major order
     *
     * No types other than unsigned 16-bit integers are known to be in
     * use for SMV.
     */
    uint16_t *raster;

    /**
     * @brief Image dimension along the vertical (slow-changing)
     *        direction, in pixels
     */
    size_t height;

    /**
     * @brief Image dimension along the horizontal (fast-changing)
     *        direction, in pixels
     */
    size_t width;

    /**
     * @brief Sample–detector distance in mm
     */
    float distance;

    /**
     * @brief Exposure time in s
     */
    float exposure;

    /**
     * @brief Oscillation range in degrees
     */
    float osc_range;

    /**
     * @brief Oscillation start in degrees
     */
    float osc_start;

    /**
     * @brief Wavelength in Å
     */
    float wavelength;
};


/**
 * @brief Allocate and initialize a new frame structure
 *
 * @return A pointer the newly allocated and initialized frame
 *         structure if successful, @c NULL otherwise.  The pointer
 *         may subsequently be used as an argument to the function
 *         frame_free(3).
 */
struct frame *
frame_new();


/**
 * @brief Free a frame structure and all its associated data
 *
 * @param frame Pointer to the frame which is to be freed
 */
void
frame_free(struct frame *frame);


/**
 * The frame_flip() function reverses the order of the pixels in each
 * column of the frame pointed to by @p frame.  The transformation is
 * performed in place.
 *
 * @param frame Pointer to the frame whose raster is to be transformed
 * @return      0 if successful, -1 otherwise.  If an error occurs,
 *              the global variable @c errno is set to indicate the
 *              error.
 */
int
frame_flip(struct frame *frame);


/**
 * The frame_rot90() function rotates the raster of the frame pointed
 * to by @p frame counterclockwise by <code>k * 90</code> degrees.
 * The transformation is performed in place.
 *
 * @param frame Pointer to the frame whose raster is to be transformed
 * @param k     Number of times the image is rotated by 90 degrees
 *              counterclockwise
 * @return      0 if successful, -1 otherwise.  If an error occurs,
 *              the global variable @c errno is set to indicate the
 *              error.
 */
int
frame_rot90(struct frame *frame, int k);


/**
 * The frame_write() function writes the frame at the location given
 * by @p frame to the stream pointed to by @p stream in SMV format.
 * Upon successful completion a value of 0 is returned.  Otherwise, a
 * value of -1 is returned and @c errno is set to:
 *
 * <dl>
 *
 *   <dt>@c EINVAL</dt><dd>The pixels are not square</dd>
 *
 *   <dt>@c EMSGSIZE</dt><dd>The @c HEADER_BYTES or the @c DATE item
 *   is longer than 512 characters</dd>
 *
 *   <dt>@c ENOMEM</dt><dd>Not enough space</dd>
 *
 * </dl>
 *
 * Any other value of @errno is due to output or encoding errors.  In
 * case of error, frame_write() may have changed the stream.
 *
 * @param frame  Pointer to the location of the frame structure
 * @param stream Pointer to the output stream
 * @return       0 if successful, -1 otherwise.  If an error occurs,
 *               the global variable @c errno is set to indicate the
 *               error.
 */
int
frame_write(const struct frame *frame, FILE *stream);

FRAME_END_C_DECLS

#endif /* !FRAME_H */
