/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * XXX Reread all the Libtiff documentation.
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <sys/mman.h>
#include <sys/stat.h>

#include <stdlib.h>

#include <errno.h>
#include <limits.h>
#include <string.h>

#include <tiffio.h>

#include "tvips.h"
#include "util.h"


/**
 * Opaque structure for maintaining I/O state on an mmap(2):d TIFF
 * stream
 */
struct tvips_tiff
{
    /* Handle to the TIFF file
     */
    TIFF *tif;

    /* Location of the mapping
     */
    void *addr;

    /* Pointer to the TemData structure
     *
     * The tem_data member points to read-only mmap(2):d memory, which
     * may not be modified.
     */
    const uint8_t *tem_data;

    /* Length of the mapping, in octets
     *
     * Since the entire stream is mmap(2):d, this is identical to the
     * length of the stream.
     */
    size_t len;

    /* File position indicator, in octets, relative to the start of
     * the stream
     */
    off_t offset;
};


uint32_t
tvips_tiff_ftoh32(struct tvips_tiff *handle, const void *file32)
{
    uint32_t host32 = *(uint32_t *)file32;
    if (TIFFIsByteSwapped(handle->tif) != 0)
        TIFFSwabLong(&host32);
    return (host32);
}


/* XXX This is completely untested!  It assumes a complex is a 2-long
 * array of real numbers.  Also, fix up parameter names like in
 * tiffio.h: *lp for ftoh32, *cp for ftohc, and *fp for ftohf.
 */
double complex
tvips_tiff_ftohc(struct tvips_tiff *handle, const void *filec)
{
    double complex hostc = *(double complex *)filec;
    if (TIFFIsByteSwapped(handle->tif) != 0)
        TIFFSwabArrayOfDouble((double *)&hostc, 2);
    return (hostc);
}


float
tvips_tiff_ftohf(struct tvips_tiff *handle, const void *filef)
{
    float hostf = *(float *)filef;
    if (TIFFIsByteSwapped(handle->tif) != 0)
        TIFFSwabFloat(&hostf);
    return (hostf);
}


/**
 * _tiff_close() is a nop: because tvips_open_tiff() does not fopen(3)
 * the file, it should not fclose(3) it either.
 */
static int
_tiff_close(thandle_t handle)
{
    return (0);
}


/**
 * _tiff_mmap() is essentially a nop: because the stream is mmap(2):d
 * in tvips_open_tiff(), it should also be munmap(2):d there.
 * _tiff_mmap() just reflects the success of mmap(2) in
 * tvips_open_tiff().
 */
static int
_tiff_mmap(thandle_t handle, tdata_t *addr, toff_t *len)
{
    struct tvips_tiff *t = handle;

    *addr = t->addr;
    *len = t->len;
    if (*addr == (void *)-1 || *len <= 0)
        return (0);
    return (1);
}


/**
 * _tiff_munmap() is a nop: because the stream is mmap(2):d in
 * tvips_open_tiff(), it should be munmap(2):d in tvips_close_tiff().
 */
static void
_tiff_munmap(thandle_t handle, tdata_t addr, toff_t len)
{}


static tsize_t
_tiff_read(thandle_t handle, tdata_t ptr, tsize_t size)
{
    struct tvips_tiff *t = handle;

    if (t->len < t->offset + size)
        size = t->len - t->offset;
    if (size > 0) {
        memcpy(ptr, t->addr + t->offset, size);
        t->offset += size;
    }
    return (size);
}


/**
 * _tiff_seek() provides lseek(2)-like behavior on an mmap(2):d
 * stream.
 */
static toff_t
_tiff_seek(thandle_t handle, toff_t offset, int whence)
{
    struct tvips_tiff *t = handle;

    switch (whence) {
    case SEEK_CUR:
        return (t->offset += offset);

    case SEEK_END:
        return (t->offset = t->len - offset);

    case SEEK_SET:
        return (t->offset = offset);
    }

    errno = EINVAL;
    return (-1);
}


static toff_t
_tiff_size(thandle_t handle)
{
    struct tvips_tiff *t = handle;

    return (t->len);
}


/**
 * _tiff_write() should never be called for read-only TIFF streams.
 */
static tsize_t
_tiff_write(thandle_t handle, tdata_t ptr, tsize_t size)
{
    return (0);
}


/**
 * The _tiff_cpy() function copies the @p w-by-@p l array of pixels
 * from array @p src to array @p dst.  The two buffers cannot overlap.
 * _tiff_cpy() will only fail if @bps is neither 8 nor 16.
 *
 * @note At the price of an increase in complexity, this function
 *       could be enhanced to support signed integer pixels and
 *       indexed images (i.e. images with a palette) as well.
 *
 * @param w   Width of block
 * @param l   Length of block
 * @param bps Bits per sample, either 8 or 16
 * @param dst Pointer to destination array of 16-bit, unsigned integer
 *            type
 * @param ldd Leading dimension of @p dst
 * @param src Pointer to source array of unsigned integer type
 * @param lds Leading dimension of @p src
 * @return    The original value of @p dst if successful, @c NULL
 *            otherwise
 */
static uint16_t *
_tiff_cpy(uint32 w,
          uint32 l,
          uint16 bps,
          uint16_t *dst,
          int32 ldd,
          const void *src,
          uint32 lds)
{
    uint32 i, j;

    switch (bps) {
    case 8:
        /* This code path is exercised by cramps.tif and
         * cramps-tile.tif.
         */
        for (i = 0; i < l; i++) {
            for (j = 0; j < w; j++)
                dst[ldd * i + j] = ((const uint8 *)src)[lds * i + j];
        }
        return (dst);

    case 16:
        /* This code path is exercised by ladoga.tif and
         * ladoga-tile.tif.
         */
        for (i = 0; i < l; i++) {
            for (j = 0; j < w; j++)
                dst[ldd * i + j] = ((const uint16 *)src)[lds * i + j];
        }
        return (dst);
    }

    return (NULL);
}


/**
 * The _tiff_raster() function asserts that image encoded in the TIFF
 * file pointed to by @p tif satisfies the constraints of the
 * implementation and returns a pointer to the decoded raster.  Sets
 * @c errno to @c ENOTSUP if the image is not supported.  Must have a
 * single image with one sample per pixel, either 8 or 16 unsigned
 * bits per pixel.  This represents the subset of TVIPS TIFF images
 * that can be losslessly represented as SMV.
 *
 * @note EMMENU prior to version XXX (not yet released) does not set
 *       the standard SAMPLEFORMAT TIFF tag properly.  These images
 *       have TIFFTAG_SAMPLEFORMAT == SAMPLEFORMAT_UINT even when the
 *       raster is composed of signed integers.  The data_type item in
 *       the TVIPS dictionary appears to be correct.
 *
 * @param[in]  tif Handle to the TIFF file
 * @param[out] w   Width of the image, in pixels
 * @param[out] l   Length of the image, in pixels
 * @param[out] bps Bits per sample, either 8 or 16
 * @return         Decoded raster, pass to free(3)
 */
static uint16_t *
_tiff_raster(TIFF *tif, uint32 *w, uint32 *l, uint16 *bps)
{
    uint16_t *raster;
    tdata_t src_data;
    tsize_t src_size;
    uint32 d, i, j, tl, tw;
    uint16 cfg, spp, sf, pm;

    raster = NULL;
    src_data = NULL;

    if (TIFFGetFieldDefaulted(
            tif, TIFFTAG_BITSPERSAMPLE, bps) == 0 || (
                *bps != 8 &&
                *bps != 16) ||
        TIFFGetFieldDefaulted(
            tif, TIFFTAG_SAMPLEFORMAT, &sf) == 0  || (
                sf != SAMPLEFORMAT_UINT &&
                sf != SAMPLEFORMAT_VOID) ||
        TIFFGetFieldDefaulted(
            tif, TIFFTAG_SAMPLESPERPIXEL, &spp) == 0 || (
                spp != 1)) {
        errno = ENOTSUP;
        return (NULL);
    }

    if (TIFFGetFieldDefaulted(
            tif, TIFFTAG_IMAGEDEPTH, &d) == 0 || (
                d != 1) ||
        TIFFGetFieldDefaulted(
            tif, TIFFTAG_IMAGELENGTH, l) == 0 ||
        TIFFGetFieldDefaulted(
            tif, TIFFTAG_IMAGEWIDTH, w) == 0) {
        errno = ENOTSUP;
        return (NULL);
    }

    if (TIFFGetFieldDefaulted(
            tif, TIFFTAG_PHOTOMETRIC, &pm) == 0 || (
                pm != PHOTOMETRIC_MINISBLACK &&
                pm != PHOTOMETRIC_MINISWHITE) ||
        TIFFGetFieldDefaulted(
            tif, TIFFTAG_PLANARCONFIG, &cfg) == 0) {
        errno = ENOTSUP;
        return (NULL);
    }

    raster = calloc(*w * *l, sizeof(uint16_t));
    if (raster == NULL)
        return (NULL);

    if (TIFFIsTiled(tif) == 0) {
        /* Striped image: read strips for sample 0 row by row, and
         * copy all the rows in the strip to the raster while
         * converting each sample on the fly.
         */
        src_size = TIFFStripSize(tif);
        src_data = _TIFFmalloc(src_size);
        if (src_data == NULL) {
            free(raster);
            return (NULL);
        }

        for (i = 0; i < *l; i += src_size / (*bps * *w / 8)) {
            src_size = TIFFReadEncodedStrip(
                tif, TIFFComputeStrip(tif, i, 0), src_data, -1);
            if (src_size == -1) {
                _TIFFfree(src_data);
                free(raster);
                return (NULL);
            }

            _tiff_cpy(
                    *w,
                    src_size / (*bps * *w / 8),
                    *bps,
                    raster + *w * i,
                    *w,
                    src_data,
                    *w);
        }
    } else {
        /* Tiled image: traverse the raster tile by tile and copy the
         * tiles while converting each sample on the fly.
         */
        if (TIFFGetFieldDefaulted(
                tif, TIFFTAG_TILELENGTH, &tl) == 0 ||
            TIFFGetFieldDefaulted(
                tif, TIFFTAG_TILEWIDTH, &tw) == 0) {
            free(raster);
            return (NULL);
        }

        src_size = TIFFTileSize(tif);
        src_data = _TIFFmalloc(src_size);
        if (src_data == NULL) {
            free(raster);
            return (NULL);
        }

        for (i = 0; i < *l; i += tl) {
            for (j = 0; j < *w; j += tw) {
                src_size = TIFFReadEncodedTile(
                    tif, TIFFComputeTile(tif, j, i, 0, 0), src_data, -1);
                if (src_size == -1) {
                    _TIFFfree(src_data);
                    free(raster);
                    return (NULL);
                }

                _tiff_cpy(
                    tw < *w - j ? tw : *w - j,
                    tl < *l - i ? tl : *l - i,
                    *bps,
                    raster + *w * i + j,
                    *w,
                    src_data,
                    tw);
            }
        }
    }

    _TIFFfree(src_data);
    return (raster);
}


const uint8_t *
tvips_tem_data_tiff(struct tvips_tiff *handle)
{
    return (handle->tem_data);
}


/* The _tiff_utf162ascii() losslessly converts a UTF-16-string to a
 * null-terminated ASCII string.  The function allocates sufficient
 * memory for a copy of the nearest ASCII-equivalent of the string
 * pointed to by @p s, does the conversion, and returns a a pointer to
 * it.  The pointer may subsequently be used as an argument to the
 * function free(3).  If insufficient memory is available, NULL is
 * returned.
 *
 * XXX What about wctob(3)?
 *
 * @param s   Pointer to an UTF-16 string
 * @param len Length of @p s in octets
 * @return    Null-terminated nearest ASCII-equivalent of @p s
 */
static char *
_tiff_utf162ascii(const void *s, size_t len)
{
    char *ascii;
    size_t i;

    ascii = calloc(len / 2 + 1, sizeof(char));
    if (ascii == NULL)
        return (NULL);

    for (i = 0; i < len / 2; i++)
        ascii[i] = ((uint16_t *)s)[i] & 0x7f;
    ascii[len / 2] = '\0';

    return (ascii);
}


/* Read the TemData structure and set errno to ENOMSG on failure.  The
 * custom TVIPS tags should be automatically read by Libtiff 3.6.0 and
 * later.
 *
 * Stuff visible in the EM_Player software on windows, for a
 * TemCam-F416, 4kx4k camera (but this is configurable from EMMENU4?):
 *
 *   Version             2
 *   Bits/Pixel         16
 *   HT [kV]           200
 *   MagTotal         2621
 *   Pixelsize [nm]  15600
 *   X Offset            0
 *   Y Offset            0
 *   X Dimension      2048
 *   Y Dimension      2048
 *   X Binning           2
 *   Y Binning           2
 */
struct tvips_tiff *
tvips_open_tiff(FILE *stream)
{
    struct stat sb;
    TIFFErrorHandler error_old, warning_old;
    struct tvips_tiff *handle;
    uint32_t *offset;
    uint32_t count, version;
    int fd;


    /* Allocate and initialize a new tvips_tiff structure such that
     * tvips_close_tiff() can be called at any point from now.
     */
    handle = malloc(sizeof(struct tvips_tiff));
    if (handle == NULL)
        return (NULL);
    handle->tif = NULL;
    handle->addr = NULL;
    handle->tem_data = NULL;
    handle->len = 0;
    handle->offset = 0;


    /* Just like the default Unix implementation in Libtiff, mmap(2)
     * the entire stream.
     */
    fd = fileno(stream);
    if (fstat(fd, &sb) != 0) {
        tvips_close_tiff(handle);
        return (NULL);
    }

    handle->offset = 0;
    handle->len = sb.st_size;
    handle->addr = mmap(
        NULL, handle->len, PROT_READ, MAP_SHARED, fd, 0);
    if (handle->addr == (void *)-1) {
        tvips_close_tiff(handle);
        return (NULL);
    }


    /* Disable errors and warnings while invoking TIFFClientOpen(), as
     * the former are communicated through errno.  Passing NULL as the
     * file name will cause TIFFClientOpen() to crash.
     */
    error_old = TIFFSetErrorHandler(NULL);
    warning_old = TIFFSetWarningHandler(NULL);
    handle->tif = TIFFClientOpen("", "r", handle,
                                 _tiff_read,
                                 _tiff_write,
                                 _tiff_seek,
                                 _tiff_close,
                                 _tiff_size,
                                 _tiff_mmap,
                                 _tiff_munmap);
    TIFFSetWarningHandler(warning_old);
    TIFFSetErrorHandler(error_old);

    if (handle->tif == NULL) {
        tvips_close_tiff(handle);
        errno = EPROTO;
        return (NULL);
    }


    /* Read the TemData structure using the new tag, which is provided
     * by EMMENU 4.0.9.25 (October 2012) and later.
     */
    if (TIFFGetField(handle->tif, 37708, &count, &handle->tem_data) == 1) {
        version = tvips_tiff_ftoh32(handle, &handle->tem_data[0]);
        if ((version == 1 && count != 244) ||
            (version == 2 && count != 6312)) {
            tvips_close_tiff(handle);
            errno = ENOMSG;
            return (NULL);
        }
        return (handle);
    }


    /* Get the value of the old tag, which is the file offset to the
     * TemData structure.
     */
    if (TIFFGetField(
            handle->tif, 37706, &count, &offset) == 1 && count == 1) {
        handle->tem_data = handle->addr + *offset;
        return (handle);
    }


    /* Disable to test non-TVIPS TIFF images.
     */
    tvips_close_tiff(handle);
    errno = ENOMSG;
    return (NULL);
}


void
tvips_close_tiff(struct tvips_tiff *handle)
{
    TIFFErrorHandler error_old, warning_old;


    if (handle == NULL)
        return;

    if (handle->tif != NULL) {
        error_old = TIFFSetErrorHandler(NULL);
        warning_old = TIFFSetWarningHandler(NULL);
        TIFFClose(handle->tif);
        TIFFSetWarningHandler(warning_old);
        TIFFSetErrorHandler(error_old);
    }

    if (handle->addr != (void *)-1 && handle->len > 0)
        munmap(handle->addr, handle->len);

    free(handle);
}


int
tvips_ctime_tiff(
    struct tvips_tiff *handle, const char *zone, struct timespec *tv)
{
    struct tm tm;
    const char *zone_old;
    time_t clock;
    uint32 tem_date, tem_time;
    int ret;


    /* If requested, temporarily adjust the timezone to match the zone
     * where the image was written.  According to Peter Sparlinek, the
     * timestamp is nowadays always recorded in localtime.
     */
    if (zone != NULL) {
        zone_old = getenv("TZ");
        if (setenv("TZ", zone, 1) != 0)
            return (-1);
        tzset();
    }

    ret = 0;
    switch (tvips_tiff_ftoh32(handle, &handle->tem_data[0])) {
    case 1:
        /* Round-trip the timestamp via localtime_r(3) and mktime(3)
         * to convert it to Unix time.  This case is completely
         * untested.
         */
        tem_date = tvips_tiff_ftoh32(handle, &handle->tem_data[224]);
        tem_time = tvips_tiff_ftoh32(handle, &handle->tem_data[228]);

        clock = tem_date * 60 * 60 * 24 + tem_time / 100;
        if (localtime_r(&clock, &tm) == NULL) {
            ret = -1;
            break;
        }

        tv->tv_sec = mktime(&tm);
        if (tv->tv_sec == -1) {
            ret = -1;
            break;
        }
        tv->tv_nsec = 10000000 * (tem_time - 100 * (tem_time / 100));
        break;


    case 2:
        /* Note that a negative value for tm_isdst causes the
         * mktime(3) function to guess whether daylight savings is in
         * effect for the given time.
         */
        tem_date = tvips_tiff_ftoh32(handle, &handle->tem_data[584]);
        tem_time = tvips_tiff_ftoh32(handle, &handle->tem_data[588]);

        tm.tm_year  = ((tem_date >> 16) & 0xffff) - 1900;
        tm.tm_mon   = ((tem_date >> 8) & 0xff) - 1;
        tm.tm_mday  = tem_date & 0xff;
        tm.tm_hour  = tem_time / (60 * 60);
        tm.tm_min   = (tem_time - tm.tm_hour * 60 * 60) / 60;
        tm.tm_sec   = tem_time - tm.tm_hour * 60 * 60 - tm.tm_min * 60;
        tm.tm_isdst = -1;

        tv->tv_sec = mktime(&tm);
        if (tv->tv_sec == -1) {
            ret = -1;
            break;
        }
        tv->tv_nsec = 0;
        break;


    default:
        /* NOTREACHED
         *
         * Disable to test non-TVIPS TIFF images.
         */
        errno = ENOTSUP;
        ret = -1;
        break;
    }


    /* Reset the timezone if it was changed.
     */
    if (zone != NULL) {
        if (zone_old != NULL)
            setenv("TZ", zone_old, 1);
        else
            unsetenv("TZ");
        tzset();
    }

    return (ret);
}


struct frame *
tvips_read_tiff(struct tvips_tiff *handle, const char *zone)
{
    uint32 binning[2];
    TIFFErrorHandler error_old, warning_old;
    struct frame *frame;
    uint32 length, width;
    uint16 bps;


    /* Disable errors and warnings, as the former are communicated
     * through errno.
     */
    error_old = TIFFSetErrorHandler(NULL);
    warning_old = TIFFSetWarningHandler(NULL);


    /* Allocate and initialize the frame structure.
     */
    frame = frame_new();
    if (frame == NULL)
        goto exit_error;

    frame->raster = _tiff_raster(handle->tif, &width, &length, &bps);
    if (frame->raster == NULL)
        goto exit_error;
    frame->width = width;
    frame->height = length;

#if 0
    {
        size_t i, j;
        for (i = 50; i < 150; i++) {
            for (j = 50; j < 50; j++)
                frame->raster[frame->width * i + j] = 0;
        }
    }
#endif


    switch (tvips_tiff_ftoh32(handle, &handle->tem_data[0])) {
    case 1:
        /* Extract the binning factor, which must be a multiple of ten
         * thousand.  Convert the exposure time in ten-thousands of ms
         * to s, pixel size in ten-thousands of µm to mm, high tension
         * in ten-thousands of kV to V, and wavelength in m to Å.
         * Note that the version 1 header assumes pixels are square
         * and equally binned in the horizontal and vertical
         * directions.
         */
        binning[0] = tvips_tiff_ftoh32(handle, &handle->tem_data[180]);
        if (binning[0] % 10000 != 0)
            goto exit_error;

        binning[0] /= 10000;
        binning[1] = binning[0];
        frame->exposure =
            1e-7 * tvips_tiff_ftoh32(handle, &handle->tem_data[196]);
        frame->pixel_size[0] =
            1e-7 * tvips_tiff_ftoh32(handle, &handle->tem_data[176]);
        frame->pixel_size[1] = frame->pixel_size[0];
        frame->wavelength = 1e10 * ht2wavelengthf(
            1e-1 * tvips_tiff_ftoh32(handle, &handle->tem_data[84]));
        break;


    case 2:
        /* Check concluding magic value.  Assert that the image
         * dimension in the TemData structure agrees with the actual
         * raster.
         */
        if (tvips_tiff_ftoh32(handle, &handle->tem_data[6308]) != 0xaaaaaaaa)
            goto exit_error;
        if (tvips_tiff_ftoh32(
                handle, &handle->tem_data[564]) != frame->width ||
            tvips_tiff_ftoh32(
                handle, &handle->tem_data[568]) != frame->height) {
            goto exit_error;
        }


        /* Assert that the pixel type can be losslessly represented by
         * the SMV format and matches the bitwidth from the TIFF
         * header.
         */
        switch (tvips_tiff_ftoh32(handle, &handle->tem_data[580])) {
        case 1:
            if (bps != 8)
                goto exit_error;
            break;

        case 2:
            if (bps != 16)
                goto exit_error;
            break;

        default:
            /* Set errno to what _tiff_raster() should have set for
             * unsupported data types.  This may happen with images
             * from EMMENU < XXX, which did not set
             * TIFFTAG_SAMPLEFORMAT correctly, thus preventing
             * _tiff_raster() from catching this error.
             */
            errno = ENOTSUP;
            goto exit_error;
        }


        /* Extract the camera type and convert it from UTF-16 to
         * ASCII.
         *
         * XXX See /groups/gonen/gonenlab/EMdata/MicroED/lysozyme/normalxtals/020414/normaldose/lmp1.tif, lets see what other strings are in that file.
         */
        frame->id = _tiff_utf162ascii(&handle->tem_data[3768], 80);
        if (frame->id == NULL)
            goto exit_error;
        if (strlen(frame->id) == 0) {
            free(frame->id);
            frame->id = NULL;
        }


        /* Extract the binning factor.  Convert the exposure time in
         * ms to s, pixel size in nm to mm, and wavelength in m to Å.
         */
        binning[0] = tvips_tiff_ftoh32(handle, &handle->tem_data[3944]);
        binning[1] = tvips_tiff_ftoh32(handle, &handle->tem_data[3948]);
        frame->exposure =
            1e3 * tvips_tiff_ftohf(handle, &handle->tem_data[3952]);
        frame->pixel_size[0] =
            1e-6 * tvips_tiff_ftohf(handle, &handle->tem_data[3928]);
        frame->pixel_size[1] =
            1e-6 * tvips_tiff_ftohf(handle, &handle->tem_data[3932]);
        frame->wavelength = 1e10 * ht2wavelengthf(
            tvips_tiff_ftohf(handle, &handle->tem_data[3272]));
        break;


    default:
        /* NOTREACHED
         *
         * Disable to test non-TVIPS TIFF images.
         */
        errno = ENOTSUP;
        goto exit_error;
    }


    /* Extract the creation time in UTC.
     */
    if (tvips_ctime_tiff(handle, zone, &frame->tv) != 0)
        goto exit_error;


    /* Convert the binning from 32-bit unsigned integer to vanilla
     * integer, considering the possibility for overflow.
     */
    if (binning[0] > UINT_MAX || binning[1] > UINT_MAX) {
        errno = EOVERFLOW;
        goto exit_error;
    }
    frame->binning[0] = binning[0];
    frame->binning[1] = binning[1];


    /* Reset the error and warning handlers.  XXX setenv(3) and
     * unsetenv(3) should not fail; if they do, they may prevent errno
     * from representing the true cause of any previous error.
     */
    TIFFSetWarningHandler(warning_old);
    TIFFSetErrorHandler(error_old);

    return (frame);


exit_error:
    /* XXX Is it possible that these calls will wreck the errno as
     * assigned above?
     */
    TIFFSetWarningHandler(warning_old);
    TIFFSetErrorHandler(error_old);

    if (frame != NULL)
        frame_free(frame);
    return (NULL);
}
