/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @todo Estimate sample--detector distance from circular solvent
 *       pattern?  Or perhaps from the (total) magnification?
 *
 * $Id:$
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include <err.h>
#include <errno.h>
#ifdef HAVE_GETOPT_H
#    include <getopt.h>
#endif
#include <libgen.h>
#include <math.h>
#include <string.h>
#include <unistd.h>

#include "frame.h"
#include "tvips.h"
#include "util.h"


struct _tv_path
{
    /* Timestamp of the file
     */
    struct timespec tv;

    /* Path of the file
     */
    const char *path;
};


/* Comparison function for _tv_path structures.  Returns an integer
 * less than, equal to, or greater than zero if the timestamp of the
 * structure pointed to by the first argument precedes, is identical,
 * or succeeds the timestamp of the structure pointed to by the second
 * argument.
 */
static int
_tv_path_compar(const void *a, const void *b)
{
    const struct timespec *ta = &(((struct _tv_path *)a)->tv);
    const struct timespec *tb = &(((struct _tv_path *)b)->tv);

    if (ta->tv_sec < tb->tv_sec)
        return (-1);
    if (ta->tv_sec > tb->tv_sec)
        return (+1);
    if (ta->tv_nsec < tb->tv_nsec)
        return (-1);
    if (ta->tv_nsec > tb->tv_nsec)
        return (+1);
    return (0);
}


/* Note that use of __progname is not portable.
 */
static void
usage()
{
    extern char *__progname;

    fprintf(stderr,
            "usage: %s "
            "[-d distance] "
            "[-g readout_geometry] "
            "[-k rotation] "
            "[-o output_template] "
            "[-r oscillation_speed] "
            "[-v] "
            "[-w wavelength] "
            "[-x beam_center_x] "
            "[-y beam_center_y] "
            "[-z timezone] "
            "file ...\n", __progname);
    fprintf(stderr,
            "       %s -V\n", __progname);
    fprintf(stderr,
            "       %s -h\n", __progname);
    exit(EXIT_FAILURE);
}


/* See the GNU coding standards for more information on this (and
 * opinions on how the --help option should have been implemented).
 */
#define xstr(s) str(s)
#define str(s) #s
static void
version()
{
    extern char *__progname;
    fprintf(stdout,
            "%s (TVIPS tools) " xstr(GIT_BRANCH) "." xstr(GIT_COMMIT) "\n",
            __progname);
}


int
main(int argc, char *argv[])
{
    float beam_center[2];
    FILE *stream;
    char *ep, *output_dir, *output_path, *path;
    const char *output_template, *zone;
    struct frame *frame;
    struct timespec *tvs;
    struct _tv_path *tv_path;
    struct tvips_tiff *handle;
    size_t j, len, nmemb;
    double exposure_max, exposure_min, t;
    float distance, exposure, tilt_rate, wavelength;
    int flip, g, i, k, k2, verbose;
    mode_t mode, mode_old;


    /* Default values for command line options.
     *
     * XXX Maybe beam_center should be -f (beam_center_fast) and -s
     * (beam_center_slow) instead of -x and -y now.  And they should
     * really default to NAN.
     */
    beam_center[0] = 32.1;
    beam_center[1] = 32.0;
    distance = 2640.0;
    g = 5;
    k = 1;
    tilt_rate = 0.09;
    output_template = NULL;
    verbose = 0;
    wavelength = NAN;
    zone = NULL;


    /* Use getopt_long(3) if available; the POSIX.2 fall-back
     * getopt(3) is assumed to always be available.  optstring begins
     * with a colon in order to enable tracking of missing option
     * arguments.
     */
    int ch;
    const char* optstring = ":Vd:g:hk:o:r:vw:x:y:z:";

#ifdef HAVE_GETOPT_LONG
    static struct option options[] = {
        { "version",          no_argument,       NULL, 'V' },
        { "distance",         required_argument, NULL, 'd' },
        { "readout-geometry", required_argument, NULL, 'g' },
        { "help",             no_argument,       NULL, 'h' },
        { "rot90",            required_argument, NULL, 'k' },
        { "output-template",  required_argument, NULL, 'o' },
        { "rotation-speed",   required_argument, NULL, 'r' },
        { "verbose",          no_argument,       NULL, 'v' },
        { "wavelength",       required_argument, NULL, 'w' },
        { "beam-center-x",    required_argument, NULL, 'x' },
        { "beam-center-y",    required_argument, NULL, 'y' },
        { "timezone",         required_argument, NULL, 'z' },
        { NULL,               0,                 NULL, 0   }
    };
#endif


    /* Loop through all the arguments in argv using either getopt(3)
     * function.  This code does its own error reporting, hence opterr
     * is set to zero to disable getopt(3) error messages.  Because
     * optreset is an extension to the POSIX.2 specification, it is
     * not used here.
     */
    opterr = 0;
#ifdef HAVE_GETOPT_LONG
    while ((ch = getopt_long(argc, argv, optstring, options, NULL)) != -1) {
#else
    while ((ch = getopt(argc, argv, optstring)) != -1) {
#endif
        switch (ch) {
        case 'V':
            version();
            exit(EXIT_SUCCESS);

        case 'd':
            errno = 0;
            distance = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(distance)) {
                warnx("Illegal -d argument %s", optarg);
                usage();
            }
            break;

        case 'g':
            errno = 0;
            g = strtol(optarg, &ep, 10);
            if (optarg[0] == '\0' || *ep != '\0' || errno != 0) {
                warnx("Illegal -g argument %s", optarg);
                usage();
            }
            break;

        case 'h':
            usage();

        case 'k':
            errno = 0;
            k = strtol(optarg, &ep, 10) % 4;
            if (optarg[0] == '\0' || *ep != '\0' || errno != 0) {
                warnx("Illegal -k argument %s", optarg);
                usage();
            }
            break;

        case 'o':
            output_template = optarg;
            break;

        case 'r':
            errno = 0;
            tilt_rate = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(tilt_rate)) {
                warnx("Illegal -r argument %s", optarg);
                usage();
            }
            break;

        case 'v':
            verbose++;
            break;

        case 'w':
            errno = 0;
            wavelength = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(wavelength)) {
                warnx("Illegal -w argument %s", optarg);
                usage();
            }
            break;

        case 'x':
            errno = 0;
            beam_center[1] = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(beam_center[1])) {
                warnx("Illegal -x argument %s", optarg);
                usage();
            }
            break;

        case 'y':
            errno = 0;
            beam_center[0] = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(beam_center[0])) {
                warnx("Illegal -y argument %s", optarg);
                usage();
            }
            break;

        case 'z':
            zone = optarg;
            break;

        case ':':
            /* Missing the required argument of an option.  Use the
             * last known option character (optopt) for error
             * reporting.
             */
#ifdef HAVE_GETOPT_LONG
            for (i = 0; options[i].name != NULL; i++) {
                if (options[i].val == optopt) {
                    warnx("Option -%c (--%s) requires an argument",
                          optopt, options[i].name);
                    usage();
                }
            }
#endif
            warnx("Option -%c requires an argument", optopt);
            usage();

        case '?':
            warnx("Unrecognized option '%s'", argv[optind - 1]);
            usage();

        default:
            usage();
        }
    }


    /* If requested, repeat command line options verbatim on standard
     * output.
     */
    if (verbose > 1) {
        version();
        printf("\n");
        for (i = 0; i < optind; i++) {
            printf("%s%s", argv[i], i + 1 < optind ? " " : "\n");
        }
        printf("\n");
    }


    /* Since the documentation states that at least on image must be
     * provided exit with failure if there are none.
     */
    if (argc <= optind)
        return (EXIT_FAILURE);
    argc -= optind;
    argv += optind;


    /* Invert any transformations applied by the camera system, then
     * apply additional rotations.
     */
    if (tvips_g2fk(g, &flip, &k2) != 0)
        err(EXIT_FAILURE, "Unsupported ReadoutGeometry %d", g);
    k += k2;


    /* First pass to estimate the exposure time, or rather, the time
     * between two successive frames, in seconds.  Would probably want
     * the standard deviation as well.  This assumes frames are
     * chronologically ordered.
     */
    tv_path = calloc(argc, sizeof(struct _tv_path));
    if (tv_path == NULL)
        err(EXIT_FAILURE, NULL);

    nmemb = 0;
    for (i = 0; i < argc; i++) {
        stream = fopen(argv[i], "r");
        if (stream == NULL) {
            free(tv_path);
            err(EXIT_FAILURE, "Failed to open %s for reading", argv[i]);
        }

        handle = tvips_open_tiff(stream);
        if (handle == NULL) {
            fclose(stream);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to read %s", argv[i]);
        }

        if (tvips_ctime_tiff(handle, zone, &tv_path[i].tv) == 0)
            tv_path[i].path = argv[i];
        else
            tv_path[i].path = NULL;
        tvips_close_tiff(handle);
        fclose(stream);
        if (tv_path[i].path == NULL) {
            free(tv_path);
            err(EXIT_FAILURE, "Failed to extract timestamp from %s", argv[i]);
        }
        nmemb += 1;


        /* If requested, output the canonicalized absolute pathname of
         * the file just read.
         */
        if (verbose > 2) {
            path = realpath(argv[i], NULL);
            if (path != NULL) {
                printf("%s\n", path);
                free(path);
            }
        }
    }


    /* Sort the input files in ascending order by timestamp.
     * Calculate exposure time and, if requested, report lower and
     * upper bounds on the time difference between successive frames.
     * XXX This assumes that successive images are uniformly
     * distributed in time.
     */
    qsort(tv_path, nmemb, sizeof(struct _tv_path), _tv_path_compar);
    tvs = calloc(nmemb, sizeof(struct timespec));
    if (tvs == NULL) {
        free(tv_path);
        err(EXIT_FAILURE, NULL);
    }
    for (j = 0; j < nmemb; j++)
        tvs[j] = tv_path[j].tv;

    exposure = deltatimef(tvs, nmemb);
    if (verbose > 0) {
        exposure_max = exposure_min = 0;
        for (j = 1; j < nmemb; j++) {
            t = difftime(tvs[j].tv_sec, tvs[j - 1].tv_sec) +
                1e-9 * (tvs[j].tv_nsec - tvs[j - 1].tv_nsec);
            if (j == 1 || t > exposure_max)
                exposure_max = t;
            if (j == 1 || t < exposure_min)
                exposure_min = t;
        }

        printf("Estimated exposure time: %.2f s "
               "(from %zd timestamp differences in the range [%.2f, %.2f] s)\n",
               exposure,
               nmemb > 0 ? nmemb - 1 : 0,
               exposure_min,
               exposure_max);
    }
    free(tvs);


    /* Second pass: traverse the input files in ascending order by
     * timestamp, apply custom alterations based on the superresolved
     * exposure time estimate, and output.  XXX Would be nice with
     * some verbose printout summarizing how much was written, number
     * of successes and failures, and so on.
     */
    if (output_template == NULL)
        return (EXIT_SUCCESS);
    len = strlen(output_template) + 1;
    output_dir = calloc(2 * len, sizeof(char));
    if (output_dir == NULL)
        err(EXIT_FAILURE, NULL);
    output_path = output_dir + len;
    mode_old = umask(0);
    mode = 0777 & ~mode_old;
    umask(mode_old);

    for (i = 0; i < nmemb; i++) {
        stream = fopen(tv_path[i].path, "r");
        if (stream == NULL) {
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to open %s for reading", tv_path[i].path);
        }

        handle = tvips_open_tiff(stream);
        if (handle == NULL) {
            fclose(stream);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to read %s", tv_path[i].path);
        }

        frame = tvips_read_tiff(handle, zone);
        tvips_close_tiff(handle);
        fclose(stream);
        if (frame == NULL) {
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to read %s", tv_path[i].path);
        }

        if (flip != 0 && frame_flip(frame) != 0) {
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to flip %s", tv_path[i].path);
        }
        if (frame_rot90(frame, k) != 0) {
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to rotate %s", tv_path[i].path);
        }


        /* The oscillation start and range is unknown for the first
         * frame.
         */
        frame->beam_center[0] = beam_center[0];
        frame->beam_center[1] = beam_center[1];
        frame->distance = distance;
        frame->exposure = i == 0 ? NAN : exposure;
        frame->osc_range = frame->exposure * tilt_rate;
        frame->osc_start = i * frame->osc_range;

        if (isfinite(wavelength))
            frame->wavelength = wavelength;


        /* Write the file to the path specified by the output
         * template, after applying "base zero" sequence number
         * substitutions and creating any intermediate directories.
         * mkpath() is always called, because the template may affect
         * the directories as well as terminal files.  Because
         * dirname(3) may wreck both its input and the output of any
         * previous invocation, both must be backed up first.
         */
        if (template2path(output_path, output_template, i) != 0) {
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to create path from template");
        }
        if (argc > 1 && errno == ERANGE && access(output_path, R_OK) != -1)
            warnx("Template truncated, %s will be overwritten", output_path);
        memmove(output_dir,
                dirname(strcpy(output_dir, output_path)),
                len * sizeof(char));
        if (mkpath(output_dir, mode) != 0) {
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to create intermediate directories");
        }

        stream = fopen(output_path, "w");
        if (stream == NULL) {
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to open %s for writing", output_path);
        }
        if (frame_write(frame, stream) != 0) {
            fclose(stream);
            frame_free(frame);
            free(output_dir);
            free(tv_path);
            err(EXIT_FAILURE, "Failed to write %s", output_path);
        }

        fclose(stream);
        frame_free(frame);
    }
    free(output_dir);
    free(tv_path);

    return (EXIT_SUCCESS);
}
