/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef TVIPS_H
#define TVIPS_H 1

#ifdef __cplusplus
#  define TVIPS_BEGIN_C_DECLS extern "C" {
#  define TVIPS_END_C_DECLS   }
#else
#  define TVIPS_BEGIN_C_DECLS
#  define TVIPS_END_C_DECLS
#endif

TVIPS_BEGIN_C_DECLS

/**
 * @file tvips.h
 * @brief XXX
 *
 * XXX Looks like the namespace is a tad inconsistent.
 */

#include <complex.h>

#include "frame.h"


/**
 * @brief Incomplete type declaration of the TVIPS TIFF handle
 */
struct tvips_tiff;


/**
 * @brief
 *
 * @param handle Pointer to an opaque TVIPS TIFF handle
 */
void
tvips_close_tiff(struct tvips_tiff *handle);


/**
 * @brief
 *
 * @param[in]  handle Pointer to an opaque TVIPS TIFF handle
 * @param[in]  zone   Timezone where the TIFF stream was written
 * @param[out] tv     Timestamp, s and ns elapsed since midnight, 1
 *                    January 1970 UTC (Unix time)
 * @return            0 if successful, -1 otherwise.  If an error
 *                    occurs, the global variable @c errno is set to
 *                    indicate the error.
 */
int
tvips_ctime_tiff(
    struct tvips_tiff *handle, const char *zone, struct timespec *tv);


/**
 * XXX
 */
double complex
tvips_tiff_ftohc(struct tvips_tiff *handle, const void *filec);


/**
 * The tvips_tiff_ftohf() function converts the single-precision
 * floating point number pointed to by @p filef from TIFF-file byte
 * ordering to the native ordering of the host.  This is necessary
 * because the TemData items are stored in TIFF-file ordering, and
 * they are not automagically converted by Libtiff.
 *
 * @param handle Handle to the TVIPS TIFF file
 * @param filef  Single-precision floating point number in TIFF-file
 *               ordering
 * @return      @p filef in native ordering of the host
 */
float
tvips_tiff_ftohf(struct tvips_tiff *handle, const void *filef);


/**
 * The tvips_tiff_ftoh32() function converts the 32-bit quantity
 * pointed to by @p file32 from TIFF-file byte ordering to the native
 * ordering of the host.  This is necessary because the TemData items
 * are stored in TIFF-file ordering, and they are not automagically
 * converted by Libtiff.
 *
 * @param handle Handle to the TVIPS TIFF file
 * @param file32 32-bit quantity in TIFF-file ordering
 * @return       @p file32 in native ordering of the host
 */
uint32_t
tvips_tiff_ftoh32(struct tvips_tiff *handle, const void *file32);


/**
 * @brief
 *
 * Set errno to EPROTO in case Libtiff fails to read the file.
 *
 * @param stream Name of input TIFF file XXX
 * @return       Pointer to an opaque TVIPS TIFF handle if successful,
 *               @c NULL otherwise.
 */
struct tvips_tiff *
tvips_open_tiff(FILE *stream);


/**
 * The tvips_read_tiff() function allocates and populates an @c frame
 * structure with data read from the seekable TIFF stream pointed to
 * by @p stream.  The timezone where the TIFF was written is given by
 * the string pointed to by @p zone.  If @p zone is @c NULL, the
 * stream is assumed to be written in the local timezone.  Upon
 * successful completion, tvips_read_tiff() returns a pointer to an
 * allocated @c frame structure.  Otherwise, @c NULL is returned and
 * the global variable @c errno is set to indicate the error.  An
 * allocated @c frame structure may subsequently be used as an
 * argument to frame_free().
 *
 * @bug XXX Is errno actually set by Libtiff?
 *
 * @note This implementation uses the Libtiff type definitions uint16,
 *       uint32, etc in lieu of the "standard" definitions from
 *       stdint.h.  Thus, the latter file need not be included.
 *
 * @param handle Pointer to an opaque TVIPS TIFF handle
 * @param zone   Timezone where the TIFF stream was written
 * @return       Allocated and populated @c frame structure
 */
struct frame *
tvips_read_tiff(struct tvips_tiff *handle, const char *zone);


/**
 * @brief
 *
 * The returned pointer is only valid until tvips_close_tiff() has
 * been called on @p handle.  XXX This will require making the
 * ftoh32() and friends public, too!
 *
 * @param handle Pointer to an opaque TVIPS TIFF handle
 * @return       Pointer to the read-only TemData structure
 */
const uint8_t *
tvips_tem_data_tiff(struct tvips_tiff *handle);


/**
 * The tvips_g2fk() function determines the column reversal and
 * rotation to invert the transformation applied by a camera system
 * configured with the given @c ReadoutGeometry parameter @p g.  If @p
 * f is non-zero the columns should be reversed using <i>e.g.</i>
 * frame_flip() prior to @p k 90-degree counterclockwise rotations
 * with <i>e.g.</i> frame_rot90().  If @p g does not define a known @c
 * ReadoutGeometry the function sets @c errno to @c EINVAL.
 *
 * @note As an extension, the undocumented readout geometries 12
 *       through 15 are supported as well.
 *
 * @bug XXX Check this documentation block with the Tietz EMMENU
 *      documentation for consistency.
 *
 * @param      g The @c ReadoutGeometry parameter of the camera system
 * @param[out] f Non-zero if the columns of the image must be reversed
 *               before rotation
 * @param[out] k The number counterclockwise 90-degree rotations that
 *               must be applied after any column reversal
 * @return       0 if successful, -1 otherwise.  If an error occurs,
 *               the global variable @c errno is set to indicate the
 *               error.
 */
int
tvips_g2fk(int g, int *f, int *k);

TVIPS_END_C_DECLS

#endif /* !TVIPS_H */
