/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * From the "Advanced recorder settings" in EMMENU4: "Images are
 * stored in image sets.  To avoid limitations of the file system,
 * image sets are only filled with a certain number of images.  If
 * that number is reached, a new image set is created automatically.
 * The file size of the image set must not exceed 2 GB.  Image sets
 * are files with the extension .tvips.  Those images can be managed
 * using the EMPlayer [sic] application."
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <err.h>
#include <errno.h>
#ifdef HAVE_GETOPT_H
#    include <getopt.h>
#endif
#include <libgen.h>
#include <limits.h>
#include <math.h>
#include <string.h>
#include <unistd.h>

#include "frame.h"
#include "tvips.h"
#include "util.h"

/* XXX Should probably use an AutoConf macro.
 */
#if defined(__linux__)
#    include <byteswap.h>
#elif defined(__APPLE__)
#    include <libkern/OSByteOrder.h>
#    define bswap_16(x) OSSwapInt16(x)
#    define bswap_32(x) OSSwapInt32(x)
#elif defined(__GNUC__)
#    define bswap_16(x) __builtin_bswap16(x)
#    define bswap_32(x) __builtin_bswap32(x)
#endif


/**
 * The _tvips_ftoh32() function converts the 32-bit quantity pointed
 * to by @p file32 from tvips-file byte ordering to the native
 * ordering of the host.  This is necessary because according to Peter
 * Sparlinek items in the TVIPS set files are always little-endian.
 * XXX This function probably only works on GCC-like compilers.
 *
 * @param file32 32-bit quantity in tvips-file ordering
 * @return       @p file32 in native ordering of the host
 */
static int32_t
_tvips_ftoh32(const void *file32)
{
    int32_t host32 = *(int32_t *)file32;
#ifdef __BIG_ENDIAN__
    host32 = bswap_32(host32);
#endif
    return (host32);
}


/**
 * The callback function must return 0 on success.  The frame
 * structure is freed using frame_free() when the callback returns.
 *
 * @param frame   Frame structure
 * @param counter Image counter, a zero-based index that continues
 *                through all the image set files
 * @param data    User data
 * @return        0 if if successful, non-zero otherwise
 */
typedef int (*tvips_frame_callback)(
    struct frame *frame, size_t counter, void *data);


/**
 * The readfile() function extracts each frame from the @p pathc set
 * files whose names are pointed to by @p pathv and, in turn, passes
 * them to @p callback.  Upon successful completion a value of 0 is
 * returned.  Otherwise, a value of -1 is returned and @c errno is set
 * to:
 *
 * <dl>
 *
 *   <dt>@c EPROTO</dt><dd>No valid series header</dd>
 *
 *   <dt>@c ENOTSUP</dt><dd>Header version not supported</dd>
 *
 *   <dt>@c EOVERFLOW</dt><dd>Binning factors out of range</dd>
 *
 * </dl>
 *
 * Any other value of @c errno is due to either failure in a function
 * from the standard library function or @p callback.
 *
 * @return      0 if successful, -1 otherwise
 */
int
readfile(size_t pathc, char *pathv[], tvips_frame_callback callback, void *data)
{
    uint8_t series_header[256];
    FILE *stream;
    size_t i, nmemb;
    int32_t lBPP, lCounter, lImgHeaderBytes, lXBin, lYBin, lXDim, lYDim;
    float lPixelSize, lWavelength;

    if (pathc-- == 0)
        return (0);

    stream = fopen(*pathv++, "r");
    if (stream == NULL)
        return (-1);


    /* Read the series header and check that lSize, the number of
     * octets in the series header structure, is 256.
     */
    if (fread(series_header, 1, 256, stream) != 256 ||
        _tvips_ftoh32(&series_header[0]) != 256) {
        fclose(stream);
        errno = EPROTO;
        return (-1);
    }


    /* Check that the version of the current file (lVersion) is
     * supported, and set up the number of octets in the image header
     * structures (lImgHeaderBytes) accordingly.
     */
    switch (_tvips_ftoh32(&series_header[4])) {
    case 1:
        lImgHeaderBytes = 12;
        break;

    case 2:
        lImgHeaderBytes = _tvips_ftoh32(&series_header[48]);
        if (lImgHeaderBytes <= 0) {
            fclose(stream);
            errno = EPROTO;
            return (-1);
        }
        break;

    default:
        fclose(stream);
        errno = ENOTSUP;
        return (-1);
    }


    /* The dimensions along x (lXDim, width) and y (lYDim, height) of
     * all the images must be positive.
     */
    lXDim = _tvips_ftoh32(&series_header[8]);
    lYDim = _tvips_ftoh32(&series_header[12]);
    if (lXDim <= 0 || lYDim <= 0) {
        fclose(stream);
        errno = EPROTO;
        return (-1);
    }


    /* The number of bits per pixel must be either 8 or 16.
     */
    lBPP = _tvips_ftoh32(&series_header[16]);
    if (lBPP != 8 && lBPP != 16) {
        fclose(stream);
        errno = EPROTO;
        return (-1);
    }


    /* Extract the binning factor and convert it from 32-bit unsigned
     * integer to vanilla integer, considering the possibility for
     * overflow.  Convert pixel size in nm to mm.
     */
    lXBin = _tvips_ftoh32(&series_header[28]);
    lYBin = _tvips_ftoh32(&series_header[32]);
    if (lXBin > UINT_MAX || lYBin > UINT_MAX) {
        fclose(stream);
        errno = EOVERFLOW;
        return (-1);
    }
    lPixelSize = 1e-6 * _tvips_ftoh32(&series_header[36]);
    lWavelength = 1e10 * ht2wavelengthf(
        1e3 * _tvips_ftoh32(&series_header[40]));


    /* The 32-bit integers lXOff and lYOff at offsets 20 and 24,
     * respectively, have no bearing on this function.  XXX Other
     * unhandled items.
     */
#if 0
    int32_t lMagTotal = _tvips_ftoh32(&series_header[44]);
#endif


    /* Read the image header.
     */
    uint8_t *image_header;
    image_header = calloc(lImgHeaderBytes, 1);
    if (image_header == NULL)
        return (-1);

    while (!feof(stream)) {
        nmemb = fread(image_header, 1, lImgHeaderBytes, stream);
        if (nmemb == 0) {
            /* Zero octets read.  Unless an error occurred, this means
             * end-of-file.
             */
            if (ferror(stream)) {
                free(image_header);
                fclose(stream);
                return (-1);
            }

            /* If there is at least one more file, open it in place of
             * the current stream.
             */
            if (pathc-- > 0) {
                fclose(stream);
                stream = fopen(*pathv++, "r");
                if (stream == NULL) {
                    free(image_header);
                    return (-1);
                }
                continue;
            }

            /* Reached end-of-file of the last stream.  Nothing more
             * to do.
             */
            break;

        } else if (nmemb == (size_t)lImgHeaderBytes) {
            /* Allocate and populate a frame structure with the image
             * data read from the file.  According to Peter Sparlinek,
             * the pixel intensities are always unsigned integers,
             * either 8 or 16 bits wide.
             */
            struct frame *frame;

            frame = frame_new();
            if (frame == NULL)
                return (-1);

            nmemb = lXDim * lYDim;
            frame->raster = calloc(nmemb, sizeof(uint16_t));
            if (frame->raster == NULL) {
                frame_free(frame);
                return (-1);
            }
            if (fread(frame->raster, lBPP / 8, nmemb, stream) != nmemb) {
                frame_free(frame);
                return (-1);
            }

            switch (lBPP) {
            case 8:
                /* Promote a raster of 8-bit unsigned integers to
                 * 16-bit dittos.  The raster need not be byteswapped.
                 * XXX This is untested code!
                 */
                for (i = nmemb; i-- > 0; )
                    frame->raster[i] = (uint16_t)((uint8_t *)frame->raster)[i];
                break;

            case 16:
                /* On big-endian hosts, a 16-bit raster must be
                 * byteswapped.  XXX This is untested code!  This
                 * function probably only works on GCC-like compilers.
                 */
#ifdef __BIG_ENDIAN__
                for (i = 0; i < nmemb; i++)
                    frame->raster[i] = bswap_16(frame->raster[i]);
#endif
                break;

            default:
                /* NOTREACHED */
                frame_free(frame);
                errno = EPROTO;
                return (-1);
            }

            /* Pass the image counter along with the populated frame
             * structure to the caller-supplied callback and free the
             * latter.  According to Peter Sparlinek, the TVIPS set
             * files give seconds (lTime) and additional milliseconds
             * (lMS) in UTC.
             */
            frame->tv.tv_sec = _tvips_ftoh32(&image_header[4]);
            frame->tv.tv_nsec = 1000000 * _tvips_ftoh32(&image_header[8]);
            frame->binning[0] = lXBin;
            frame->binning[1] = lXBin;
            frame->pixel_size[0] = lPixelSize;
            frame->pixel_size[1] = lPixelSize;
            frame->width = lYDim;
            frame->height = lXDim;
            frame->wavelength = lWavelength;

            lCounter = _tvips_ftoh32(&image_header[0]);
            if (lCounter < 1) {
                frame_free(frame);
                errno = EPROTO;
                return (-1);
            }
            if (callback(frame, lCounter - 1, data) != 0) {
                frame_free(frame);
                return (-1);
            }
            frame_free(frame);
        } else {
            /* Unexpected object count.  This always indicates an
             * error.
             */
            free(image_header);
            fclose(stream);
            errno = EPROTO;
            return (-1);
        }
    }

    free(image_header);
    fclose(stream);

    return (0);
}


struct output
{
    struct timespec *tvs;
    const char *output_template;
    float beam_center[2];
    size_t capacity;
    size_t nmemb;
    float distance;
    float tilt_rate;
    float wavelength;
    int k;
    int flip;
};


/**
 * @bug XXX This function should (but does not) set @c errno properly.
 *
 * @param counter Image counter, a zero-based index that continues
 *                through all the image set files
 */
static int
_frame_write(struct frame *frame, size_t counter, void *data)
{
    struct output *output;
    FILE *stream;
    char *output_dir, *output_path;
    void *p;
    size_t len;
    double exposure, t;
    mode_t mode, mode_old;
    int ret;


    /* If counter does not match the number of frames actually seen,
     * return with EPROTO.  In what follows, this function uses
     * output->nmemb in lieu of counter.
     */
    output = (struct output *)data;
    if (output->nmemb != counter) {
        errno = EPROTO;
        return (-1);
    }


    /* Append the timestamp of the frame to the list of timestamps for
     * bookkeeping.  The list is dynamically reallocated, if
     * necessary.
     */
    if (output->nmemb >= output->capacity) {
        p = realloc(
            output->tvs, (output->capacity + 1024) * sizeof(struct timespec));
        if (p == NULL)
            return (-1);
        output->tvs = p;
        output->capacity += 1024;
    }
    output->tvs[output->nmemb] = frame->tv;


    /* The oscillation start and range is unknown for the first frame.
     * Calculate the time since the first exposure.  XXX The first
     * exposure is assumed to be the first frame--it is assumed that
     * frames are chronologically ordered.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    if (output->nmemb == 0) {
        exposure = NAN;
        t = NAN;
    } else {
        exposure = difftime(frame->tv.tv_sec,
                            output->tvs[output->nmemb - 1].tv_sec) +
            1e-9 * (frame->tv.tv_nsec -
                    output->tvs[output->nmemb - 1].tv_nsec);
        t = difftime(frame->tv.tv_sec, output->tvs[0].tv_sec) +
            1e-9 * (frame->tv.tv_nsec - output->tvs[0].tv_nsec);
    }

    output->nmemb += 1;


    /* No further action if no output is requested.
     */
    if (output->output_template == NULL)
        return (0);


    /* Complete the frame structure with items that are (currently)
     * not extracted from the file headers but supplied on the command
     * line.  If the wavelength is supplied on the command line, it
     * overrides the value extracted from the file headers.  Apply the
     * frame rotation.
     */
    frame->beam_center[0] = output->beam_center[0];
    frame->beam_center[1] = output->beam_center[1];
    frame->distance = output->distance;
    frame->exposure = exposure;
    frame->osc_range = exposure * output->tilt_rate;
    frame->osc_start = t * output->tilt_rate;

    if (isfinite(output->wavelength))
        frame->wavelength = output->wavelength;

    if (output->flip != 0 && frame_flip(frame) != 0)
        return (-1);
    if (frame_rot90(frame, output->k) != 0)
        return (-1);


    /* Write the file to the path specified by the output template,
     * after applying "base zero" sequence number substitutions and
     * creating any intermediate directories.  mkpath() is always
     * called, because the template may affect the directories as well
     * as terminal files.  Because dirname(3) may wreck both its input
     * and the output of any previous invocation, both must be backed
     * up first.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    len = strlen(output->output_template) + 1;
    output_dir = calloc(2 * len, sizeof(char));
    if (output_dir == NULL)
        return (-1);
    output_path = output_dir + len;
    mode_old = umask(0);
    mode = 0777 & ~mode_old;
    umask(mode_old);

    if (template2path(
            output_path, output->output_template, output->nmemb - 1) != 0) {
        free(output_dir);
        return (-1);
    }
    memmove(output_dir,
            dirname(strcpy(output_dir, output_path)),
            len * sizeof(char));
    if (mkpath(output_dir, mode) != 0) {
        free(output_dir);
        return (-1);
    }

    stream = fopen(output_path, "w");
    free(output_dir);
    if (stream == NULL)
        return (-1);

    ret = frame_write(frame, stream);
    fclose(stream);
    if (ret != 0)
        return (-1);
    return (0);
}


/* Note that use of __progname is not portable.
 *
 * XXX This is duplication w.r.t. tiff2smv.c.
 */
static void
usage()
{
    extern char *__progname;

    fprintf(stderr,
            "usage: %s "
            "[-d distance] "
            "[-g readout_geometry] "
            "[-k rotation] "
            "[-o output_template] "
            "[-r oscillation_speed] "
            "[-v] "
            "[-w wavelength] "
            "[-x beam_center_x] "
            "[-y beam_center_y] "
            "[-z timezone] "
            "file ...\n", __progname);
    fprintf(stderr,
            "       %s -V\n", __progname);
    fprintf(stderr,
            "       %s -h\n", __progname);
    exit(EXIT_FAILURE);
}


/* See the GNU coding standards for more information on this (and
 * opinions on how the --help option should have been implemented).
 *
 * XXX This is duplication w.r.t. tiff2smv.c.
 */
#define xstr(s) str(s)
#define str(s) #s
static void
version()
{
    extern char *__progname;
    fprintf(stdout,
            "%s (TVIPS tools) " xstr(GIT_BRANCH) "." xstr(GIT_COMMIT) "\n",
            __progname);
}


int
main(int argc, char *argv[])
{
#if 0
    size_t i;
    for (i = 1; i < argc; i++)
        readfile(1, &argv[i], my_callback, NULL);
#else
    struct output output;
    char *ep, *path;
    size_t j;
    double exposure_max, exposure_min, t;
    float exposure;
    int g, i, k, verbose;


    /* Default values for command line options.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    output.tvs = NULL;
    output.beam_center[0] = 32.1;
    output.beam_center[1] = 32.0;
    output.nmemb = 0;
    output.capacity = 0;
    output.distance = 2640.0;
    output.tilt_rate = 0.09;
    output.output_template = NULL;
    output.wavelength = NAN;

    g = 5;
    k = 1;
    verbose = 0;


    /* Use getopt_long(3) if available; the POSIX.2 fall-back
     * getopt(3) is assumed to always be available.  optstring begins
     * with a colon in order to enable tracking of missing option
     * arguments.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    int ch;
    const char* optstring = ":Vd:g:hk:o:r:vw:x:y:z:";

#ifdef HAVE_GETOPT_LONG
    static struct option options[] = {
        { "version",          no_argument,       NULL, 'V' },
        { "distance",         required_argument, NULL, 'd' },
        { "readout-geometry", required_argument, NULL, 'g' },
        { "help",             no_argument,       NULL, 'h' },
        { "rot90",            required_argument, NULL, 'k' },
        { "output-template",  required_argument, NULL, 'o' },
        { "rotation-speed",   required_argument, NULL, 'r' },
        { "verbose",          no_argument,       NULL, 'v' },
        { "wavelength",       required_argument, NULL, 'w' },
        { "beam-center-x",    required_argument, NULL, 'x' },
        { "beam-center-y",    required_argument, NULL, 'y' },
        { "timezone",         required_argument, NULL, 'z' },
        { NULL,               0,                 NULL, 0   }
    };
#endif


    /* Loop through all the arguments in argv using either getopt(3)
     * function.  This code does its own error reporting, hence opterr
     * is set to zero to disable getopt(3) error messages.  Because
     * optreset is an extension to the POSIX.2 specification, it is
     * not used here.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    opterr = 0;
#ifdef HAVE_GETOPT_LONG
    while ((ch = getopt_long(argc, argv, optstring, options, NULL)) != -1) {
#else
    while ((ch = getopt(argc, argv, optstring)) != -1) {
#endif
        switch (ch) {
        case 'V':
            version();
            exit(EXIT_SUCCESS);

        case 'd':
            errno = 0;
            output.distance = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(output.distance)) {
                warnx("Illegal -d argument %s", optarg);
                usage();
            }
            break;

        case 'g':
            errno = 0;
            g = strtol(optarg, &ep, 10);
            if (optarg[0] == '\0' || *ep != '\0' || errno != 0) {
                warnx("Illegal -g argument %s", optarg);
                usage();
            }
            break;

        case 'h':
            usage();

        case 'k':
            errno = 0;
            k = strtol(optarg, &ep, 10) % 4;
            if (optarg[0] == '\0' || *ep != '\0' || errno != 0) {
                warnx("Illegal -k argument %s", optarg);
                usage();
            }
            break;

        case 'o':
            output.output_template = optarg;
            break;

        case 'r':
            errno = 0;
            output.tilt_rate = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(output.tilt_rate)) {
                warnx("Illegal -r argument %s", optarg);
                usage();
            }
            break;

        case 'v':
            verbose++;
            break;

        case 'w':
            errno = 0;
            output.wavelength = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(output.wavelength)) {
                warnx("Illegal -w argument %s", optarg);
                usage();
            }
            break;

        case 'x':
            errno = 0;
            output.beam_center[1] = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(output.beam_center[1])) {
                warnx("Illegal -x argument %s", optarg);
                usage();
            }
            break;

        case 'y':
            errno = 0;
            output.beam_center[0] = strtof(optarg, &ep);
            if (optarg[0] == '\0' ||
                *ep != '\0' ||
                errno != 0 ||
                !isfinite(output.beam_center[0])) {
                warnx("Illegal -y argument %s", optarg);
                usage();
            }
            break;

        case 'z':
            /* This is useless for .tvips because timestamps are
             * always in UTC.
             */
            warnx("Timezone ignored for TVIPS set files");
            break;

        case ':':
            /* Missing the required argument of an option.  Use the
             * last known option character (optopt) for error
             * reporting.
             */
#ifdef HAVE_GETOPT_LONG
            for (i = 0; options[i].name != NULL; i++) {
                if (options[i].val == optopt) {
                    warnx("Option -%c (--%s) requires an argument",
                          optopt, options[i].name);
                    usage();
                }
            }
#endif
            warnx("Option -%c requires an argument", optopt);
            usage();

        case '?':
            warnx("Unrecognized option '%s'", argv[optind - 1]);
            usage();

        default:
            usage();
        }
    }


    /* If requested, repeat command line options verbatim on standard
     * output.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    if (verbose > 1) {
        version();
        printf("\n");
        for (i = 0; i < optind; i++) {
            printf("%s%s", argv[i], i + 1 < optind ? " " : "\n");
        }
        printf("\n");
    }


    /* Since the documentation states that at least on image must be
     * provided exit with failure if there are none.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    if (argc <= optind)
        return (EXIT_FAILURE);
    argc -= optind;
    argv += optind;


    /* Invert any transformations applied by the camera system, then
     * apply additional rotations.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    if (tvips_g2fk(g, &output.flip, &output.k) != 0)
        err(EXIT_FAILURE, "Unsupported ReadoutGeometry %d", g);
    output.k += k;


    /* XXX Validation suggestion: sort the streams by timestamp of the
     * first frame in the stream.  Then verify that lCounter is
     * continuous.
     */

    /* Convert and output the frames.  This assumes that we see the
     * images in chronological order.  XXX Would be nice with some
     * verbose printout summarizing how much was written, number of
     * successes and failures, and so on.
     */
    if (readfile(argc, argv, _frame_write, (void *)&output) != 0) {
        if (output.tvs != NULL)
            free(output.tvs);
        err(EXIT_FAILURE, "Failed to convert");
    }


    /* If requested, output the canonicalized absolute pathname of the
     * files just read.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    if (verbose > 2) {
        for (i = 0; i < argc; i++) {
            path = realpath(argv[i], NULL);
            if (path != NULL) {
                printf("%s\n", path);
                free(path);
            }
        }
    }


    /* Calculate exposure time and, if requested, report lower and
     * upper bounds on the time difference between successive frames.
     *
     * XXX This is duplication w.r.t. tiff2smv.c.
     */
    exposure = deltatimef(output.tvs, output.nmemb);
    if (verbose > 0) {
        exposure_max = exposure_min = 0;
        for (j = 1; j < output.nmemb; j++) {
            t = difftime(output.tvs[j].tv_sec, output.tvs[j - 1].tv_sec) +
                1e-9 * (output.tvs[j].tv_nsec - output.tvs[j - 1].tv_nsec);
            if (j == 1 || t > exposure_max)
                exposure_max = t;
            if (j == 1 || t < exposure_min)
                exposure_min = t;
        }

        printf("Estimated exposure time: %.2f s "
               "(from %zd timestamp differences in the range [%.2f, %.2f] s)\n",
               exposure,
               output.nmemb > 0 ? output.nmemb - 1 : 0,
               exposure_min,
               exposure_max);
    }

    if (output.tvs != NULL)
        free(output.tvs);
#endif

    return (EXIT_SUCCESS);
}
