/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <libgen.h>
#include <string.h>

#include "util.h"


/* This recursive implementation is preferred over the iterative
 * implementation of mkpath() in OpenBSD's mkdir(1) command, because
 * it avoids hard-coding the directory separator.
 */
int
mkpath(const char *path, mode_t mode)
{
    struct stat sb;
    char *parent, *path_bak;

    /* Recursively create non-existent parent directory.  Need to back
     * up path, because dirname(3) may change it.
     */
    path_bak = strdup(path);
    parent = dirname(path_bak);
    if (stat(parent, &sb) != 0 &&
        mkpath(parent, mode | S_IWUSR | S_IXUSR) != 0) {
        free(path_bak);
        return (-1);
    }
    free(path_bak);

    if (stat(path, &sb) == 0) {
        if (!S_ISDIR(sb.st_mode)) {
            /* Path exists, but is not a directory.
             */
            errno = ENOTDIR;
            return (-1);
        }
    } else {
        /* Create path with requested permissions.
         */
        if (mkdir(path, mode) != 0)
            return (-1);
    }

    /* Ensure terminal directory has the requested file mode.
     */
    if (mode > 0777 && chmod(path, mode) != 0)
        return (-1);
    return (0);
}


int
template2path(char *dst, const char *src, size_t seqno)
{
    size_t i, j;
    int d, ret;

    /* ret < 0 signifies no substitution has been attempted, ret > 0
     * means at least one substitution was truncated.  Unless ret == 0
     * this function should set errno to ERANGE.
     */
    ret = -1;

    for (i = 0; src[i] != '\0'; i++) {
        /* Base case: straight copy from src to dst.
         */
        if (src[i] != '#') {
            dst[i] = src[i];
            continue;
        }

        /* Substitution case: determine length of the template, print
         * a left-justified textual representation of the sequence
         * number, including the final '\0', into the template field.
         */
        if (ret < 0)
            ret = 0;
        for (j = 1; src[i + j] == '#'; j++)
            ;

        d = snprintf(dst + i, j + 1, "%zd", seqno);
        if (d == -1) {
            return (-1);

        } else if (d > j) {
            /* The sequence number did not fit into the field.  This
             * error is not fatal.
             */
            ret = 1;

        } else if (d < j) {
            /* The sequence number did not completely fill the field.
             * Right-justify and zero-pad.
             */
            memmove(dst + i + j - d, dst + i, d);
            for (d = i + j - d - 1; d >= i; d--)
                dst[d] = '0';
        }

        i += j - 1;
    }

    if (ret != 0)
        errno = ERANGE;
    return (0);
}
