/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil; tab-width: 8 -*- */

/*-
 * Copyright (c) 2015, Howard Hughes Medical Institute
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef UTIL_H
#define UTIL_H 1

#ifdef __cplusplus
#  define UTIL_BEGIN_C_DECLS extern "C" {
#  define UTIL_END_C_DECLS   }
#else
#  define UTIL_BEGIN_C_DECLS
#  define UTIL_END_C_DECLS
#endif

UTIL_BEGIN_C_DECLS

/**
 * @file util.h
 * @brief XXX
 *
 * XXX
 */

#include <sys/stat.h>

#include <time.h>


/**
 * @brief Determine interval of successive time points
 *
 * The deltatime() function determines the time difference between the
 * @p nmemb elements of an array of @c timespec structures.  The
 * elements of the array must be sorted on entry.  Assuming the
 * elements are evenly spaced, the return value is optimal in the
 * least-squares sense.
 *
 * @param base  Initial member of an ordered @p nmemb-long array of @c
 *              timespec structures
 * @param nmemb Length of @p base
 * @return      The time difference between successive elements of @p
 *              base
 */
double
deltatime(const struct timespec *base, size_t nmemb);


/**
 * @copydoc deltatime()
 */
float
deltatimef(const struct timespec *base, size_t nmemb);


/**
 * @brief de Broglie wavelength of an accelerated electron
 *
 * The ht2wavelength() function calculates the de Broglie wavelength
 * of an electron under an accelerating electrostatic potential @p ht.
 * If @p ht is non-positive, ht2wavelength() returns zero and sets the
 * global variable @c errno to @c EDOM.
 *
 * @param ht High tension, the accelerating potential, in V
 * @return   Wavelength, in m
 */
double
ht2wavelength(double ht);


/**
 * @copydoc ht2wavelength()
 */
float
ht2wavelengthf(float ht);


/**
 * The mkpath() function creates a directory and any intermediate
 * directories as required.  Intermediate directories are always
 * created with write and search permissions for the owner.
 *
 * @param path Path of new, terminal directory
 * @param mode File mode of terminal directory
 * @return     0 if successful, -1 otherwise.  If an error occurs, the
 *             global variable @c errno is set to indicate the error.
 */
int
mkpath(const char *path, mode_t mode);


/**
 * @brief Copy string with template substitution
 *
 * The template2path() function copies a template string, pointed to
 * by @p src, to @p dst and substitutes all consecutive stretches of
 * number signs ('#') with a textual representation of @p seqno.  If
 * the length of any such stretch is less than the length of the
 * textual representation of @p seqno the number is right-truncated.
 * If truncation occurs or @p src does not contain any characters to
 * be substituted template2path() stores @c ERANGE in @c errno before
 * returning successfully.
 *
 * @bug @p dst and @p src should probably be declared with the @c
 *      restrict keyword.
 *
 * @param dst   Pointer to output path string, at least as long as @p
 *              template
 * @param src   Pointer to template path string
 * @param seqno Non-negative, integer sequence number
 * @return      0 if successful, -1 otherwise.  If an error occurs,
 *              the global variable @c errno is set to indicate the
 *              error.
 */
int
template2path(char *dst, const char *src, size_t seqno);

UTIL_END_C_DECLS

#endif /* !UTIL_H */
